import React, { Component } from 'react';
import electron from 'electron';
import cn from 'classnames';
import path from 'path';
import fillTemplate from 'es6-dynamic-template';
import Interweave from 'interweave';
import { withStyles } from '@material-ui/core/styles';
import storage from 'azure-storage';
import moment from 'moment';
import { History } from 'history';
import { Logger } from 'library/Logger';
import badgeIcon from 'resources/images/fail.svg';
import mainIcon from 'resources/images/cardo_unit.svg';
import {
    SUPPORT_URL,
    AZURE_BLOB_SERVICE_CONNECTION_STRING,
    AZURE_BLOB_SERVICE_LOGS_CONTAINER_NAME,
    AZURE_BLOB_SERVICE_LOGS_URL
} from 'constants/urls';
import routes from 'constants/routes';
import StyledButton from 'components/StyledButton';
import styles from './styles';

type Props = {
    language: string,
    strings: object,
    history: History,
    classes: {
        [key: string]: string
    }
};

let tryAgainBtn;
let backHomeBtn;

class ProblemUpdatingFirmwareScreen extends Component<Props> {
    constructor(props) {
        super(props);
        this.state = {
            language: props.language,
            strings: props.strings
        };
    }

    static getDerivedStateFromProps(nextProps, prevState) {
        if (prevState.language !== nextProps.language) {
            return { language: nextProps.language, strings: nextProps.strings };
        }
        return null;
    }

    componentDidMount() {
        Logger.info('ProblemUpdatingFirmwareScreen: Mounted');

        this.sendLogs();

        tryAgainBtn = document.getElementById('tryAgainBtn');
        backHomeBtn = document.getElementById('backToMainBtn');

        this.checkButtonSize();
    }

    componentDidUpdate() {
        this.checkButtonSize();
    }

    checkButtonSize() {
        backHomeBtn.style.width = 'auto';
        tryAgainBtn.style.width = 'auto';
        // 42 here is the size of paddings and border so width applying for inner content
        if (tryAgainBtn.offsetWidth > backHomeBtn.offsetWidth) {
            backHomeBtn.style.width = `${tryAgainBtn.offsetWidth - 42}px`;
        } else {
            tryAgainBtn.style.width = `${backHomeBtn.offsetWidth - 42}px`;
        }
    }

    onBackToMain() {
        const { history } = this.props;
        // Move to main screen, but do not perform auto update checks again
        history.replace({
            pathname: routes.WELCOME.path,
            state: { skipAutoUpdate: true }
        });
    }

    onTryAgain() {
        const { history } = this.props;
        // Move to update device screen
        history.replace(routes.UPDATE_DEVICE.path);
    }

    async sendLogs() {
        // Send over the log files
        const logDir = electron.remote.app.getPath('userData');
        const appLogPath = path.join(logDir, 'updater.log');
        const libraryLogPath = path.join(logDir, 'cfu.log');

        const blobService = storage.createBlobService(
            AZURE_BLOB_SERVICE_CONNECTION_STRING
        );
        const blockBlobContainerName = AZURE_BLOB_SERVICE_LOGS_CONTAINER_NAME;

        const currentTime = moment().format('YYYY-MM-DD_HHmmss');
        const uuid =
            Math.random()
                .toString(36)
                .substring(2, 15) +
            Math.random()
                .toString(36)
                .substring(2, 15);

        const appLogFileBlobName = `${currentTime}-${uuid}-app.log`;
        const libraryLogFileBlobName = `${currentTime}-${uuid}-cfu.log`;

        console.log('Uploading app log file', appLogFileBlobName);

        blobService.createBlockBlobFromLocalFile(
            blockBlobContainerName,
            appLogFileBlobName,
            appLogPath,
            error => {
                console.log('Finished uploading app log file', error);
            }
        );

        console.log('Uploading library log file', libraryLogFileBlobName);

        blobService.createBlockBlobFromLocalFile(
            blockBlobContainerName,
            libraryLogFileBlobName,
            libraryLogPath,
            error => {
                console.log('Finished uploading library log file', error);
            }
        );

        const baseUrl = `${AZURE_BLOB_SERVICE_LOGS_URL}/${AZURE_BLOB_SERVICE_LOGS_CONTAINER_NAME}`;

        console.log('Base URL', baseUrl);
    }

    render() {
        const { strings, language } = this.state;
        const { classes } = this.props;
        return (
            <div className={classes.container}>
                <div className={classes.innerContainer}>
                    <div className={classes.imageContainer}>
                        <img
                            className={classes.mainImage}
                            src={mainIcon}
                            alt="Main Icon"
                        />
                        <img
                            className={classes.badgeImage}
                            src={badgeIcon}
                            alt="Badge Icon"
                        />
                    </div>
                    <div className={classes.title}>
                        {strings.problem_updating_fw_title}
                    </div>
                    <div className={classes.subtitle}>
                        {' '}
                        {strings.problem_updating_fw_subtitle}
                    </div>
                    <div className={classes.buttonContainer}>
                        <StyledButton
                            id="tryAgainBtn"
                            variant="contained"
                            btnSize="big"
                            onClick={() => this.onTryAgain()}
                            styles={cn(
                                classes.tryAgain,
                                classes[`tryAgain-${language}`]
                            )}
                        >
                            {strings.general_button_retry}
                        </StyledButton>
                        <StyledButton
                            id="backToMainBtn"
                            variant="outlined"
                            btnSize="big"
                            onClick={() => this.onBackToMain()}
                            styles={cn(
                                classes.backToMain,
                                classes[`backToMain-${language}`]
                            )}
                        >
                            {strings.problem_updating_fw_back_to_main_button}
                        </StyledButton>
                    </div>
                    <div className={classes.help}>
                        <Interweave
                            content={fillTemplate(
                                strings.problem_updating_fw_help,
                                { url: SUPPORT_URL }
                            )}
                        />
                    </div>
                </div>
            </div>
        );
    }
}

export default withStyles(styles)(ProblemUpdatingFirmwareScreen);
